<?php

namespace App\Http\Controllers;

use App\Models\Event;
use App\Models\EventRegistration;
use App\Models\Feedback;
use App\Models\Memorandum;
use App\Models\Poll;
use App\Models\PollOption;
use App\Models\Vote;
use Illuminate\Http\Request;

class PublicEngagementController extends Controller
{
    public function index()
    {
        // Stats data
        $stats = [
            'feedbackCount' => Feedback::count(),
            'usersCount' => \App\Models\User::count(),
            'reformsCount' => 42, // Replace with actual count from your model
            'newLawsCount' => 15,
            'amendmentsCount' => 28,
            'simplifiedCount' => 42,
        ];

        // Recent feedback topics
        $recentTopics = Feedback::latest()->get();

        // Active calls for memoranda
        // $activeCalls = Memorandum::latest()->get();
        $activeCalls = Memorandum::where('deadline', '>=', now())
            ->orderBy('deadline')
            ->take(3)
            ->get()
            ->map(function ($memorandum) {
                return [
                    'id' => $memorandum->id,
                    'title' => $memorandum->title,
                    'status' => $memorandum->deadline > now() ? 'Open' : 'Closed',
                    'status_color' => $memorandum->deadline > now() ? 'primary' : 'success',
                    'deadline_iso' => $memorandum->deadline->toIsoString(),
                    'deadline_formatted' => $memorandum->deadline->format('F j, Y'),
                    'description' => $memorandum->description
                ];
            })
            ->toArray();
        // Polls data
        $currentPoll = Poll::with(['options' => function ($query) {
            $query->withCount('votes');
        }])
            ->where('is_active', true)
            ->where('start_date', '<=', now()->format('Y-m-d H:i:s'))
            ->where('end_date', '>=', now()->format('Y-m-d H:i:s'))
            // ->where('start_date', '==', now())
            // ->where('end_date', '>=', now())
            ->first();

        if ($currentPoll) {
            // Calculate total votes
            $currentPoll->total_votes = $currentPoll->options->sum('votes_count');
            $currentPoll->days_left = now()->diffInDays($currentPoll->end_date);
        }

        $recentPolls = Poll::with(['options' => function ($query) {
            $query->withCount('votes');
        }])
            ->where('end_date', '<', now())
            ->orderBy('end_date', 'desc')
            ->take(2)
            ->get()
            ->each(function ($poll) {
                $poll->total_votes = $poll->options->sum('votes_count');
                $poll->options->each(function ($option) use ($poll) {
                    $option->percentage = $poll->total_votes > 0
                        ? round(($option->votes_count / $poll->total_votes) * 100)
                        : 0;
                });
            });
        $influentialPolls = \App\Models\InfluentialPoll::with('poll')
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get();


        // Events data
        $upcomingEvents = Event::where('end_date', '>', now())
            ->where('is_published', true)
            ->orderBy('end_date')
            ->get()
            ->each(function ($event) {
                $event->date_time = $event->start_date->format('F j, Y | h:i A');
                $event->status = $event->supporters > 50 ? 'Limited Space' : 'Available';
                $event->status_color = $event->supporters > 50 ? 'warning' : 'primary';
            });


        // Testimonials
        // $testimonials = [
        //     [
        //         'name' => 'Aisha Balogun',
        //         'role' => 'Small Business Owner',
        //         'quote' => 'My feedback on business regulations was actually incorporated into the new law. It\'s empowering to see our voices make a real difference in governance.',
        //         'image' => 'https://randomuser.me/api/portraits/women/45.jpg'
        //     ],
        //     [
        //         'name' => 'Chinedu Okoro',
        //         'role' => 'Environmental Activist',
        //         'quote' => 'The public hearing on environmental laws was well organized. Commissioners genuinely listened to our concerns and asked thoughtful follow-up questions.',
        //         'image' => 'https://randomuser.me/api/portraits/men/32.jpg'
        //     ],
        //     [
        //         'name' => 'Funke Adebayo',
        //         'role' => 'Legal Researcher',
        //         'quote' => 'Submitting a memorandum was straightforward. I received confirmation and updates throughout the process. Transparency at its best!',
        //         'image' => 'https://randomuser.me/api/portraits/women/68.jpg'
        //     ]
        // ];

        return view('public-engagement', array_merge($stats, [
            'recentTopics' => $recentTopics,
            'activeCalls' => $activeCalls,
            'currentPoll' => $currentPoll,
            'recentPolls' => $recentPolls,
            'influentialPolls' => $influentialPolls,
            'upcomingEvents' => $upcomingEvents,
            // 'testimonials' => $testimonials
        ]));
    }

    public function submitFeedback(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'category' => 'required|string',
            'message' => 'required|string'
        ]);

        // Add IP address
        $validated['ip_address'] = $request->ip();

        Feedback::create($validated);

        // return back()->with('success', 'Thank you for your feedback!');
        return back()
            ->with('success', 'Thank you for your feedback!, We are glad that stopped by')
            ->with('active_tab', $request->active_tab);
    }

    public function submitMemorandum(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'title' => 'required|string|max:255',
            'legislation' => 'required|string',
            'document' => 'required|file|mimes:pdf,doc,docx|max:5120',
            'summary' => 'required|string',
            'deadline' => 'required|date|after:today' // Add this validation
        ]);

        // Handle file upload
        $path = $request->file('document')->store('memoranda');

        // Create memorandum record
        Memorandum::create(array_merge($validated, [
            'document_path' => $path,
            'ip_address' => $request->ip() // Add IP address
        ]));

        // return back()->with('success', 'Memorandum submitted successfully!');
        return back()
            ->with('success', 'Your Memorandum submitted successfully, please make sure to work with the deadline provided!')
            ->with('active_tab', $request->active_tab);
    }

    public function voteOnPoll(Request $request, Poll $poll)
    {
        $request->validate([
            'option' => 'required|exists:poll_options,id'
        ]);

        $userIp = $request->ip();

        // Check if user has already voted
        if (session()->has("voted_{$poll->id}") || $poll->votes()->where('user_ip', $userIp)->exists()) {
            // return back()->with('error', 'You have already voted on this poll.');
            return back()
                ->with('error', 'You have already voted on this poll.')
                ->with('active_tab', $request->active_tab);
        }

        // Record vote
        Vote::create([
            'poll_id' => $poll->id,
            'option_id' => $request->option,
            'user_ip' => $userIp
        ]);

        // Increment the vote count for the selected option
        \App\Models\PollOption::where('id', $request->option)->increment('votes');

        // Store voting status in session
        session()->put("voted_{$poll->id}", true);

        // return back()->with('success', 'Vote recorded successfully!');
        return back()
            ->with('success', 'Vote recorded successfully!, please note that you can only vote once per poll')
            ->with('active_tab', $request->active_tab);
    }


    public function supportEvent(Event $event)
    {
        $event->increment('supporters');
        return response()->json(['success' => true, 'supporters' => $event->supporters]);
    }

    public function registerForEvent(Request $request)
    {
        $validated = $request->validate([
            'event_id' => 'required|exists:events,id',
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'organization' => 'nullable|string|max:255',
            'attending_as' => 'required|string',
            'requirements' => 'nullable|string'
        ]);

        EventRegistration::create($validated);

        // return back()->with('success', 'Registration successful!');
        return back()
            ->with('success', 'Your Event Registration was successful!!,')
            ->with('active_tab', $request->active_tab);
    }
}
