<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ReformProjectResource\Pages;
use App\Models\ReformProject;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class ReformProjectResource extends Resource
{
    protected static ?string $model = ReformProject::class;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Textarea::make('description')
                    ->required()
                    ->columnSpanFull(),
                Select::make('focus_area')
                    ->options([
                        'Gender' => 'Gender',
                        'Criminal Justice' => 'Criminal Justice',
                        'Constitutional' => 'Constitutional',
                        'Others' => 'Others',
                    ])
                    ->required(),
                Select::make('status')
                    ->options([
                        'Planning' => 'Planning',
                        'In Progress' => 'In progress',
                        'Completed' => 'Completed',
                    ])
                    ->required(),
                DatePicker::make('start_date'),
                DatePicker::make('end_date'),
                FileUpload::make('progress_report')
                    ->directory('reform-projects/reports')
                    ->acceptedFileTypes(['application/pdf'])
                    ->downloadable()
                    ->openable()
                    ->disk('public'),
                Textarea::make('outcomes')
                    ->columnSpanFull(),
                Hidden::make('user_id')
                    ->default(Auth::id())
                    ->visibleOn('create'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('focus_area')
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'Planning' => 'info',
                        'In Progress' => 'warning',
                        'Completed' => 'success',
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('start_date')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('end_date')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Created by')
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('focus_area')
                    ->options([
                        'Gender' => 'Gender',
                        'Criminal Justice' => 'Criminal Justice',
                        'Constitutional' => 'Constitutional',
                        'Others' => 'Others',
                    ]),
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'Planning' => 'Planning',
                        'In Progress' => 'In progress',
                        'Completed' => 'Completed',
                    ])
                    ->indicator(fn(string $value): string => match ($value) {
                        'Planning' => 'Planning',
                        'In Progress' => 'In Progress',
                        'Completed' => 'Completed',
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];

                        if ($data['value'] ?? null) {
                            $indicators[] = Tables\Filters\Indicator::make($data['value'])
                                ->color(match ($data['value']) {
                                    'Planning' => 'info',
                                    'In Progress' => 'warning',
                                    'Completed' => 'success',
                                });
                        }

                        return $indicators;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make(),
                Tables\Actions\DeleteAction::make()->successNotification(Notification::make()->success()->title("Law Reform Project Deleted.")->body('The Law Reform Project Deleted Successfully')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->where('user_id', Auth::id());
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListReformProjects::route('/'),
            'create' => Pages\CreateReformProject::route('/create'),
            'edit' => Pages\EditReformProject::route('/{record}/edit'),
        ];
    }

    public static function getNavigationGroup(): ?string
    {
        return 'Content Management';
    }
}
