<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PollResource\Pages;
use App\Models\Poll;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\HasOne;
use Filament\Forms\Components\Section;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class PollResource extends Resource
{
    protected static ?string $model = Poll::class;

    protected static ?string $navigationIcon = 'heroicon-o-chart-bar';
    protected static ?string $navigationGroup = 'Public Engagement';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('title')
                    ->required()
                    ->maxLength(255)
                    ->columnSpanFull(),

                Textarea::make('description')
                    ->required()
                    ->columnSpanFull(),

                Grid::make(2)
                    ->schema([
                        DateTimePicker::make('start_date')->required(),
                        DateTimePicker::make('end_date')->required(),
                    ]),

                Toggle::make('is_active')
                    ->default(true)
                    ->inline(false),

                Repeater::make('options')
                    ->relationship('options') // Make sure this exists in Poll.php
                    ->schema([
                        TextInput::make('text')
                            ->required()
                            ->maxLength(255),
                    ])
                    ->defaultItems(2)
                    ->minItems(2)
                    ->reorderable()
                    ->addActionLabel('Add Option')
                    ->columnSpanFull(),

                Section::make('Poll Impact')
                    ->relationship('influentialPoll') // This tells Filament which relationship to load
                    ->schema([
                        TextInput::make('title')->label('Impact Title')->required(),
                        Textarea::make('impact_description')->rows(2)->required(),
                    ])
                    ->collapsed()
                    ->collapsible(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')->searchable()->limit(40),

                TextColumn::make('start_date')->dateTime()->sortable(),
                TextColumn::make('end_date')->dateTime()->sortable(),

                BadgeColumn::make('status')
                    ->getStateUsing(function (Poll $record) {
                        if (!$record->start_date || !$record->end_date) return 'Unknown';

                        if ($record->end_date < now()) return 'Closed';
                        if ($record->start_date > now()) return 'Scheduled';
                        return 'Active';
                    })
                    ->colors([
                        'success' => 'Active',
                        'warning' => 'Scheduled',
                        'danger' => 'Closed',
                    ]),

                IconColumn::make('is_active')->boolean()->sortable()->label('Published'),

                // TextColumn::make('total_votes')
                //     ->label('Votes')
                //     ->getStateUsing(fn(Poll $record) => $record->options()->sum('votes')),

                TextColumn::make('total_votes')
                    ->label('Votes')
                    ->getStateUsing(fn(Poll $record) => $record->votes()->count()),

                IconColumn::make('has_impact')
                    ->label('Impact')
                    ->boolean()
                    ->getStateUsing(fn($record) => $record->influentialPoll()->exists()),
            ])
            ->filters([])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPolls::route('/'),
            'create' => Pages\CreatePoll::route('/create'),
            'edit' => Pages\EditPoll::route('/{record}/edit'),
        ];
    }
}
